import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { 
  Wrench, 
  CheckCircle, 
  Phone, 
  Mail,
  Lightbulb,
  Ruler,
  Palette,
  Settings,
  Building2,
  Target
} from "lucide-react";
import Link from "next/link";

export default function CustomSolutionsPage() {
  const customServices = [
    {
      title: "Size Modifications",
      description: "Custom dimensions to fit specific space requirements",
      icon: <Ruler className="h-8 w-8" />,
      examples: ["Extended lengths", "Non-standard widths", "Height adjustments", "Multi-level designs"]
    },
    {
      title: "Interior Layouts",
      description: "Tailored interior configurations for specific use cases", 
      icon: <Building2 className="h-8 w-8" />,
      examples: ["Open plan offices", "Partitioned rooms", "Reception areas", "Storage integration"]
    },
    {
      title: "Special Features",
      description: "Additional functionality and specialized equipment",
      icon: <Settings className="h-8 w-8" />,
      examples: ["Server rooms", "Laboratory setups", "Medical facilities", "Clean rooms"]
    },
    {
      title: "Aesthetic Options",
      description: "Custom finishes and branding to match your requirements",
      icon: <Palette className="h-8 w-8" />,
      examples: ["Corporate colors", "Logo integration", "Premium finishes", "Architectural styling"]
    }
  ];

  const customProjects = [
    {
      name: "Modified Container Office",
      description: "20ft shipping container converted to executive office",
      features: ["Executive interiors", "Climate control", "Reception area", "Meeting room"],
      application: "Corporate site office"
    },
    {
      name: "Double Story Porta Cabin",
      description: "Two-level cabin for maximum space utilization",
      features: ["Ground floor office", "Upper level conference", "External staircase", "Structural reinforcement"],
      application: "Large construction projects"
    },
    {
      name: "Medical Facility Cabin",
      description: "Specialized cabin for healthcare applications",
      features: ["Sterile surfaces", "Medical gas points", "Isolation rooms", "Emergency access"],
      application: "Temporary medical facilities"
    },
    {
      name: "Multi-Unit Complex",
      description: "Connected cabin complex for comprehensive facilities",
      features: ["Multiple departments", "Covered walkways", "Central utilities", "Expandable design"],
      application: "Industrial operations"
    }
  ];

  const customizationProcess = [
    {
      step: "01",
      title: "Consultation",
      description: "Discuss your requirements and site conditions with our experts"
    },
    {
      step: "02", 
      title: "Design",
      description: "Create detailed plans and 3D visualizations of your custom solution"
    },
    {
      step: "03",
      title: "Approval",
      description: "Review and approve the final design, specifications, and quotation"
    },
    {
      step: "04",
      title: "Manufacturing",
      description: "Build your custom cabin in our Sharjah facility with quality controls"
    },
    {
      step: "05",
      title: "Installation",
      description: "Professional delivery and installation at your location across UAE"
    }
  ];

  const capabilities = [
    "Structural modifications and reinforcements",
    "Electrical system customization",
    "Plumbing and HVAC integration",
    "Security system installation",
    "Fire safety and compliance",
    "Accessibility features (disabled access)",
    "Environmental controls and monitoring",
    "Communication and IT infrastructure",
    "Specialized equipment mounting",
    "Multi-unit interconnection systems"
  ];

  return (
    <div className="min-h-screen">
      {/* Hero Section */}
      <section className="relative py-20 bg-gradient-to-r from-purple-900/80 to-blue-900/80">
        <div className="absolute inset-0 bg-black/40"></div>
        <div className="absolute inset-0">
          <div className="w-full h-full bg-gradient-to-br from-purple-800 via-blue-800 to-slate-800"></div>
        </div>
        
        <div className="relative z-20 container mx-auto px-4 text-center text-white">
          <h1 className="text-4xl md:text-6xl font-bold mb-6">
            Custom Solutions
          </h1>
          <p className="text-xl md:text-2xl max-w-3xl mx-auto">
            Tailored Porta Cabin Solutions for Unique Requirements
          </p>
        </div>
      </section>

      {/* Custom Services */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Customization Services
            </h2>
            <p className="text-lg text-gray-700 max-w-3xl mx-auto">
              Every project is unique. Our custom solutions service allows you to modify standard designs 
              or create completely new porta cabin configurations to meet your specific requirements.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8 mb-16">
            {customServices.map((service, index) => (
              <Card key={index} className="text-center hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="bg-purple-100 w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-4">
                    <div className="text-purple-600">
                      {service.icon}
                    </div>
                  </div>
                  <CardTitle className="text-xl font-bold">{service.title}</CardTitle>
                  <CardDescription>{service.description}</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    {service.examples.map((example, idx) => (
                      <div key={idx} className="flex items-center text-sm text-gray-600">
                        <CheckCircle className="h-3 w-3 mr-2 text-purple-600" />
                        <span>{example}</span>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Custom Projects Showcase */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
              Custom Project Examples
            </h2>
            <p className="text-lg text-gray-600">
              See how we've created unique solutions for diverse client requirements
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mb-16">
            {customProjects.map((project, index) => (
              <Card key={index} className="overflow-hidden hover:shadow-lg transition-shadow">
                <div className="relative h-48 bg-gradient-to-br from-purple-800 to-blue-800">
                  <Badge className="absolute top-4 left-4 bg-purple-600">
                    Custom Project
                  </Badge>
                  <Badge className="absolute top-4 right-4 bg-white text-gray-800">
                    {project.application}
                  </Badge>
                </div>
                
                <CardHeader>
                  <CardTitle className="text-xl font-bold">{project.name}</CardTitle>
                  <CardDescription>{project.description}</CardDescription>
                </CardHeader>
                
                <CardContent>
                  <div className="space-y-2">
                    <h4 className="text-sm font-semibold text-gray-900 mb-2">Custom Features:</h4>
                    {project.features.map((feature, idx) => (
                      <div key={idx} className="flex items-center text-sm text-gray-600">
                        <CheckCircle className="h-3 w-3 mr-2 text-purple-600" />
                        <span>{feature}</span>
                      </div>
                    ))}
                  </div>
                  
                  <Button className="w-full mt-4 bg-purple-600 hover:bg-purple-700">
                    <Lightbulb className="mr-2 h-4 w-4" />
                    Discuss Similar Project
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Customization Process */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
              Our Customization Process
            </h2>
            <p className="text-lg text-gray-600">
              From concept to completion, we guide you through every step
            </p>
          </div>

          <div className="max-w-4xl mx-auto">
            <div className="grid grid-cols-1 md:grid-cols-5 gap-8">
              {customizationProcess.map((process, index) => (
                <div key={index} className="text-center">
                  <div className="bg-purple-600 text-white w-12 h-12 rounded-full flex items-center justify-center mx-auto mb-4 text-lg font-bold">
                    {process.step}
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">
                    {process.title}
                  </h3>
                  <p className="text-gray-600 text-sm">
                    {process.description}
                  </p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Capabilities */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-3xl font-bold text-gray-900 mb-6 flex items-center">
                <Wrench className="h-8 w-8 mr-3 text-purple-600" />
                Technical Capabilities
              </h2>
              <p className="text-lg text-gray-700 mb-6">
                Our engineering team and manufacturing facility can handle complex customizations 
                while maintaining quality and safety standards.
              </p>
              
              <div className="space-y-3">
                {capabilities.map((capability, index) => (
                  <div key={index} className="flex items-start space-x-3">
                    <CheckCircle className="h-5 w-5 text-purple-600 mt-0.5 flex-shrink-0" />
                    <span className="text-gray-700">{capability}</span>
                  </div>
                ))}
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <Card className="p-6 text-center bg-purple-50">
                <Target className="h-12 w-12 text-purple-600 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Precision</h3>
                <p className="text-gray-600">Exact specifications and quality control</p>
              </Card>
              <Card className="p-6 text-center bg-blue-50">
                <Lightbulb className="h-12 w-12 text-blue-600 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Innovation</h3>
                <p className="text-gray-600">Creative solutions for unique challenges</p>
              </Card>
              <Card className="p-6 text-center bg-green-50">
                <Building2 className="h-12 w-12 text-green-600 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Integration</h3>
                <p className="text-gray-600">Seamless system integration</p>
              </Card>
              <Card className="p-6 text-center bg-orange-50">
                <Settings className="h-12 w-12 text-orange-600 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Support</h3>
                <p className="text-gray-600">Ongoing technical support</p>
              </Card>
            </div>
          </div>
        </div>
      </section>

      {/* Call to Action */}
      <section className="py-16 bg-purple-800 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready for a Custom Solution?
          </h2>
          <p className="text-xl mb-8 max-w-2xl mx-auto">
            Let's discuss your unique requirements. Our team will work with you to create 
            the perfect porta cabin solution for your specific needs.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-white text-purple-800 hover:bg-gray-100 px-8">
              <Phone className="mr-2 h-5 w-5" />
              Call: +971 58 201 2073
            </Button>
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white/10 px-8" asChild>
              <Link href="/contact">
                <Mail className="mr-2 h-5 w-5" />
                Start Custom Project
              </Link>
            </Button>
          </div>
        </div>
      </section>
    </div>
  );
}